# hyperliquid_ws.py
import asyncio
import websockets
import json
import os
import pytz
import time
from datetime import datetime, timedelta
from lib.config_manager import ConfigManager
# Add this import with other imports
from lib.historical_storage import HistoricalStorage
import ntplib
from bot.telegram_bot import generate_summary, generate_detailed_dollar_report, send_telegram_report, set_historical_storage

# ==================== Storage Path ====================
STORAGE_DIR = "/home/vabotsss/public_html/bot1/storage"

# ==================== Time Frame Config Function ====================
config = ConfigManager()

# ==================== Historical Storage Initialization ====================
historical_storage = HistoricalStorage()
set_historical_storage(historical_storage)
# ============================================================================

def get_time_frame_config():
    base_config = {
        "5m": ("5 Minutes", 300, "⚪"),
        "15m": ("15 Minutes", 900, "🔵"),
        "30m": ("30 Minutes", 1800, "🟣"),
        "45m": ("45 Minutes", 2700, "🟠"),
        "1h": ("1 Hour", 3600, "🟢"),
        "2h": ("2 Hours", 7200, "🟤"),
        "4h": ("4 Hours", 14400, "🟡"),
        "6h": ("6 Hours", 21600, "🟣"),
        "8h": ("8 Hours", 28800, "🟠"),
        "12h": ("12 Hours", 43200, "🟠"),
        "1d": ("1 Day", 86400, "🔴"),
        "3d": ("3 Days", 259200, "🟤"),
        "1w": ("1 Week", 604800, "🔵")
    }
    
    time_frames = config.TIME_FRAMES
    result = {}
    
    for tf in time_frames:
        if tf in base_config:
            result[tf] = base_config[tf]
        else:
            try:
                if tf.endswith('m'):
                    minutes = int(tf[:-1])
                    seconds = minutes * 60
                    result[tf] = (f"{minutes} Minutes", seconds, "⚪")
                elif tf.endswith('h'):
                    hours = int(tf[:-1])
                    seconds = hours * 3600
                    result[tf] = (f"{hours} Hour{'s' if hours > 1 else ''}", seconds, "🟢")
                elif tf.endswith('d'):
                    days = int(tf[:-1])
                    seconds = days * 86400
                    result[tf] = (f"{days} Day{'s' if days > 1 else ''}", seconds, "🔴")
                elif tf.endswith('w'):
                    weeks = int(tf[:-1])
                    seconds = weeks * 604800
                    result[tf] = (f"{weeks} Week{'s' if weeks > 1 else ''}", seconds, "🔵")
            except ValueError:
                print(f"Warning: Invalid time frame: {tf}")
    
    return result
# ===================================================================

# ==================== Color Settings ====================
ANSI_COLORS = {
    "yellow": "\033[93m",
    "purple": "\033[95m",
    "blue": "\033[94m",
    "red": "\033[91m",
    "green": "\033[92m",
    "cyan": "\033[96m",
    "white": "\033[97m",
}
RESET_COLOR = "\033[0m"

coin_colors = {}
for coin, color in config.COIN_COLORS.items():
    if color in ANSI_COLORS:
        coin_colors[coin] = ANSI_COLORS[color]
# =====================================================

DATA_FILE = os.path.join(STORAGE_DIR, "trades.json")
STATE_FILE = os.path.join(STORAGE_DIR, "state.json")

# Global storage for all trades meeting the dollar threshold
all_significant_trades = []
whale_trades = []
cum_d_volumes = {symbol: 0.0 for symbol in config.SYMBOLS}
last_clean_time = time.time()
last_save_time = time.time()
stored_day = None
reporting_task = None

# Dollar volume data structure
dollar_volume_data = {}
for timeframe in config.DOLLAR_VOLUME_TIMEFRAMES:
    dollar_volume_data[timeframe] = {"start_time": 0, "data": {}}

# Add these global variables
last_reports = {}
report_cooldown = 60  # 60 seconds cooldown between identical reports

def load_state():
    try:
        if os.path.exists(STATE_FILE):
            with open(STATE_FILE, 'r') as f:
                return json.load(f)
    except Exception as e:
        print(f"Error loading state: {str(e)}")
    return None

def save_state():
    try:
        os.makedirs(STORAGE_DIR, exist_ok=True)
        temp_path = os.path.join(STORAGE_DIR, "state.json.tmp")
        state = {
            "last_update": time.time(),
            "cum_d_volumes": cum_d_volumes,
            "stored_day": stored_day
        }
        with open(temp_path, 'w') as f:
            json.dump(state, f)
        os.replace(temp_path, STATE_FILE)
    except Exception as e:
        print(f"Error saving state: {str(e)}")

def load_existing_trades():
    try:
        if os.path.exists(DATA_FILE):
            with open(DATA_FILE, "r") as f:
                return json.load(f)
        return []
    except Exception as e:
        print(f"Error loading trades: {str(e)}")
        return []

def save_trades():
    try:
        os.makedirs(STORAGE_DIR, exist_ok=True)
        temp_path = os.path.join(STORAGE_DIR, "trades.json.tmp")
        
        # Debug logging
        print(f"Saving {len(whale_trades)} trades to {temp_path}")
        
        with open(temp_path, "w") as f:
            json.dump(whale_trades, f)
        
        # Debug logging
        print(f"Replacing {temp_path} with {DATA_FILE}")
        
        os.replace(temp_path, DATA_FILE)
        
        # Debug logging
        print(f"Successfully saved {len(whale_trades)} trades")
        
    except Exception as e:
        print(f"Error saving trades: {str(e)}")
        import traceback
        traceback.print_exc()

def clean_old_trades():
    global whale_trades, all_significant_trades
    current_time = get_ntp_time()
    cutoff = current_time - (config.MAX_TRADE_AGE_DAYS * 86400)
    
    # Clean whale trades
    initial_count = len(whale_trades)
    whale_trades = [t for t in whale_trades if t["timestamp"] >= cutoff]
    
    # Clean all significant trades
    initial_significant_count = len(all_significant_trades)
    all_significant_trades = [t for t in all_significant_trades if t["timestamp"] >= cutoff]
    
    if initial_count != len(whale_trades) or initial_significant_count != len(all_significant_trades):
        print(f"Cleaned {initial_count - len(whale_trades)} whale trades and {initial_significant_count - len(all_significant_trades)} significant trades")
        save_trades()

def get_ntp_time():
    try:
        ntp_client = ntplib.NTPClient()
        for server in ['pool.ntp.org', 'time.google.com']:
            try:
                response = ntp_client.request(server)
                return response.tx_time
            except:
                continue
        return datetime.now(pytz.timezone("UTC")).timestamp()
    except:
        return datetime.now(pytz.timezone("UTC")).timestamp()

def calculate_cumulative_volumes(trades, symbol, day_start):
    cum_d = 0.0
    
    for trade in trades:
        if trade["coin"] != symbol:
            continue
            
        signed_vol = trade["volume"] if trade["side"] == "B" else -trade["volume"]
        
        if trade["timestamp"] >= day_start:
            cum_d += signed_vol
            
    return cum_d

def get_candle_range(current_timestamp, seconds, reporting_interval):
    current_candle_end = ((current_timestamp // seconds) + 1) * seconds
    current_candle_start = current_candle_end - seconds
    
    prev_candle_end = current_candle_start
    prev_candle_start = prev_candle_end - seconds
    
    time_since_prev_candle_close = current_timestamp - prev_candle_end
    
    if current_timestamp >= prev_candle_end and time_since_prev_candle_close <= reporting_interval:
        return prev_candle_start, prev_candle_end, prev_candle_end
    else:
        return current_candle_start, current_candle_end, current_timestamp

# New updated
def update_dollar_volumes(coin, dollar_value, side, timestamp):
    """Update dollar volume data for all configured timeframes"""
    global dollar_volume_data
    
    for timeframe in config.DOLLAR_VOLUME_TIMEFRAMES:
        if timeframe not in dollar_volume_data:
            dollar_volume_data[timeframe] = {"start_time": 0, "data": {}}
        
        timeframe_seconds = get_timeframe_seconds(timeframe)
        timeframe_start = timestamp - (timestamp % timeframe_seconds)
        
        # Reset data if we've moved to a new timeframe
        if timeframe_start != dollar_volume_data[timeframe]["start_time"]:
            # Save the completed timeframe before resetting
            if dollar_volume_data[timeframe]["start_time"] != 0:
                historical_storage.save_dollar_volume_state(
                    timeframe, 
                    dollar_volume_data[timeframe]["start_time"],
                    dollar_volume_data[timeframe]["data"]
                )
            
            dollar_volume_data[timeframe] = {"start_time": timeframe_start, "data": {}}
        
        if coin not in dollar_volume_data[timeframe]["data"]:
            dollar_volume_data[timeframe]["data"][coin] = {"buy": 0.0, "sell": 0.0}
        
        if side == "B":
            dollar_volume_data[timeframe]["data"][coin]["buy"] += dollar_value
        else:
            dollar_volume_data[timeframe]["data"][coin]["sell"] += dollar_value
        
        # Save state after each significant update
        if dollar_value > config.BASE_DOLLAR_THRESHOLD:
            historical_storage.save_dollar_volume_state(
                timeframe, 
                dollar_volume_data[timeframe]["start_time"],
                dollar_volume_data[timeframe]["data"]
            )
# =================================================================
def get_timeframe_seconds(timeframe):
    """Convert timeframe string to seconds"""
    if timeframe.endswith('m'):
        return int(timeframe[:-1]) * 60
    elif timeframe.endswith('h'):
        return int(timeframe[:-1]) * 3600
    elif timeframe.endswith('d'):
        return int(timeframe[:-1]) * 86400
    return 900  # Default to 15 minutes

def should_display_trade(coin, dollar_value):
    """Determine if a trade should be displayed based on adaptive threshold"""
    # Priority coins use base threshold
    if coin in config.PRIORITY_COINS:
        return dollar_value >= config.BASE_DOLLAR_THRESHOLD
    
    # Mini priority coins use mini threshold
    if coin in config.MINI_PRIORITY_COINS:
        return dollar_value >= config.MINI_DOLLAR_THRESHOLD
    
    # Normal coins use base threshold
    return dollar_value >= config.BASE_DOLLAR_THRESHOLD


def generate_dollar_volume_data(timeframe, current_time):
    """Generate dollar volume data for a specific timeframe based on all significant trades"""
    global all_significant_trades
    
    if not all_significant_trades:
        print(f"⚠️ No significant trades found for {timeframe} dollar volume report")
        return {}
    
    # Get timeframe configuration
    timeframe_config = get_time_frame_config()
    if timeframe not in timeframe_config:
        print(f"⚠️ Invalid timeframe for dollar volume: {timeframe}")
        return {}
    
    seconds = timeframe_config[timeframe][1]
    reporting_interval = config.INTERVAL_MINUTES * 60
    
    # Use the same candle range calculation as whale reports
    candle_start, candle_end, trade_end = get_candle_range(
        current_time, 
        seconds,
        reporting_interval
    )
    
    # Filter trades within this candle range
    relevant_trades = [
        t for t in all_significant_trades 
        if candle_start <= t["timestamp"] < trade_end
    ]
    
    # بقیه تابع بدون تغییر...
    if not relevant_trades:
        print(f"⚠️ No trades found in {timeframe} timeframe ({datetime.fromtimestamp(candle_start)} to {datetime.fromtimestamp(candle_end)})")
        return {}
    
    print(f"📊 Generating {timeframe} dollar volume report with {len(relevant_trades)} trades")
    
    # Aggregate data by coin
    coin_data = {}
    for trade in relevant_trades:
        coin = trade["coin"]
        if coin not in coin_data:
            coin_data[coin] = {"buy": 0.0, "sell": 0.0}
        
        if trade["side"] == "B":
            coin_data[coin]["buy"] += trade["dollar_value"]
        else:
            coin_data[coin]["sell"] += trade["dollar_value"]
    
    # Filter out coins below threshold
    filtered_coin_data = {}
    for coin, volumes in coin_data.items():
        total_volume = volumes["buy"] + volumes["sell"]
        if total_volume >= config.BASE_DOLLAR_THRESHOLD:
            filtered_coin_data[coin] = volumes
        else:
            print(f"🔻 Skipping {coin} in {timeframe}: ${total_volume:,.0f} < ${config.BASE_DOLLAR_THRESHOLD:,.0f}")
    
    # Debug output
    if filtered_coin_data:
        print(f"✅ {timeframe} dollar volume data: {len(filtered_coin_data)} coins meet threshold")
        for coin, data in filtered_coin_data.items():
            print(f"   {coin}: Buy=${data['buy']:,.0f}, Sell=${data['sell']:,.0f}")
    else:
        print(f"⚠️ No coins meet threshold for {timeframe} dollar volume report")
    
    return filtered_coin_data

async def process_trade_data(data):
    global whale_trades, cum_d_volumes, all_significant_trades, last_save_time
    
    try:
        data = json.loads(data)
        if data.get("channel") == "trades" and data.get("data"):
            current_time = get_ntp_time()
            
            for trade in data["data"]:
                coin = trade.get("coin")
                
                try:
                    volume = float(trade.get("sz", 0))
                except ValueError:
                    volume = 0.0
                
                try:
                    timestamp = trade.get("time", current_time * 1000) / 1000
                except (TypeError, ValueError):
                    timestamp = current_time
                
                try:
                    price = float(trade.get("px", 0))
                except (TypeError, ValueError):
                    price = 0.0
                
                side = trade.get("side", "unknown")
                dollar_value = volume * price
                
                # Debug print for significant dollar volume trades
                if dollar_value >= config.BASE_DOLLAR_THRESHOLD:
                    print(f"💵 Significant dollar trade: {coin} ${dollar_value:,.0f} ({side})")
                
                # Update dollar volumes for all coins with significant volume
                if (config.REPORT_DOLLAR_VOLUMES and dollar_value >= config.BASE_DOLLAR_THRESHOLD and 
    coin in config.DOLLAR_SYMBOLS):
                    update_dollar_volumes(coin, dollar_value, side, timestamp)
                    # Also add to all_significant_trades for dollar volume reporting
                    all_significant_trades.append({
                        "timestamp": timestamp,
                        "dollar_value": dollar_value,
                        "side": side,
                        "coin": coin,
                        "volume": volume,
                        "price": price
                    })
                
                # Only process whale trades for configured symbols
                if coin in config.SYMBOLS:
                    threshold = config.THRESHOLDS.get(coin, 1.0)
                    
                    if volume >= threshold:
                        signed_volume = volume if side == "B" else -volume
                        cum_d_volumes[coin] = cum_d_volumes.get(coin, 0.0) + signed_volume
                        
                        # Check if should display based on adaptive threshold
                        should_display = True
                        if config.ADAPTIVE_THRESHOLD_ENABLED:
                            should_display = should_display_trade(coin, dollar_value)
                        
                        if should_display:
                            trade_data = {
                                "timestamp": timestamp,
                                "volume": volume,
                                "side": side,
                                "price": price,
                                "coin": coin,
                                "dollar_value": dollar_value
                            }
                            whale_trades.append(trade_data)
                            
                            side_str = "Buy" if side == "B" else "Sell"
                            color_code = coin_colors.get(coin, RESET_COLOR)
                            
                            print(
                                f"{color_code}{coin:8}  {side_str:5} {volume:12.2f}  "
                                f"{price:12.4f}$   "
                                f"CumD:{cum_d_volumes.get(coin, 0.0):+14.2f}"
                                f" | ${dollar_value:,.0f}{RESET_COLOR}"
                            )
                            
                            # Save trades after processing new ones - ADD THIS CODE
                            if current_time - last_save_time > 60:  # Save at most once per minute
                                try:
                                    save_trades()
                                    last_save_time = current_time
                                except Exception as e:
                                    print(f"Error saving after trade processing: {e}")
                            
    except Exception as e:
        print(f"Error processing trade data: {str(e)}")
        import traceback
        traceback.print_exc()

async def reporting_loop():
    global last_clean_time, stored_day, cum_d_volumes, last_reports, last_save_time
    
    time_frame_config = get_time_frame_config()
    
    interval_seconds = config.INTERVAL_MINUTES * 60
    current_time = get_ntp_time()
    next_report_time = (current_time // interval_seconds + 1) * interval_seconds
    wait_time = next_report_time - current_time
    
    if wait_time > 0:
        print(f"Waiting {wait_time:.1f} seconds for first report...")
        await asyncio.sleep(wait_time)
    
    consecutive_errors = 0
    
    while True:
        try:
            current_time = get_ntp_time()
            current_utc_day = datetime.utcfromtimestamp(current_time).strftime("%Y-%m-%d")
            
            if stored_day != current_utc_day:
                if stored_day is not None:
                    print(f"Resetting daily volumes for new day: {current_utc_day}")
                    for symbol in config.SYMBOLS:
                        cum_d_volumes[symbol] = 0.0
                stored_day = current_utc_day
                save_state()
            
            if current_time - last_clean_time > 86400:
                clean_old_trades()
                last_clean_time = current_time
            
            # Periodic save every 5 minutes
            if current_time - last_save_time > 300:
                try:
                    save_trades()
                    
                    # NEW: Save dollar volume state for persistence
                    if config.REPORT_DOLLAR_VOLUMES:
                        for timeframe in config.DOLLAR_VOLUME_TIMEFRAMES:
                            if timeframe in dollar_volume_data:
                                historical_storage.save_dollar_volume_state(
                                    timeframe,
                                    dollar_volume_data[timeframe]["start_time"],
                                    dollar_volume_data[timeframe]["data"]
                                )
                    
                    last_save_time = current_time
                    print("Periodic save completed")
                except Exception as e:
                    print(f"Error during periodic save: {e}")
            
            # First send individual symbol reports
            for symbol in config.SYMBOLS:
                symbol_trades = [t for t in whale_trades if t["coin"] == symbol]
                
                if not symbol_trades:
                    continue
                
                # Collect candle data for historical storage (without affecting current reports)
                for tf, (label, seconds, bullet) in time_frame_config.items():
                    current_timestamp = current_time
                    candle_start, candle_end, trade_end = get_candle_range(
                        current_timestamp, seconds, interval_seconds
                    )
                    
                    relevant_trades = [
                        t for t in symbol_trades
                        if candle_start <= t["timestamp"] < trade_end
                    ]
                    
                    # Calculate metrics for historical storage
                    buy_trades = [t for t in relevant_trades if t.get("side") == "B"]
                    sell_trades = [t for t in relevant_trades if t.get("side") != "B"]
                    
                    buy_volume = sum(t["volume"] for t in buy_trades)
                    sell_volume = sum(t["volume"] for t in sell_trades)
                    total_volume = buy_volume + sell_volume
                    
                    avg_price = 0
                    if total_volume > 0:
                        avg_price = sum(t.get("price", 0) * t["volume"] for t in relevant_trades) / total_volume
                    
                    close_price = 0
                    if relevant_trades:
                        last_trade = max(relevant_trades, key=lambda x: x["timestamp"])
                        close_price = last_trade["price"]
                    
                    max_buy_volume = 0
                    max_buy_price = 0
                    if buy_trades:
                        max_buy_trade = max(buy_trades, key=lambda x: x["volume"])
                        max_buy_volume = max_buy_trade["volume"]
                        max_buy_price = max_buy_trade["price"]
                    
                    max_sell_volume = 0
                    max_sell_price = 0
                    if sell_trades:
                        max_sell_trade = max(sell_trades, key=lambda x: x["volume"])
                        max_sell_volume = max_sell_trade["volume"]
                        max_sell_price = max_sell_trade["price"]
                    
                    # Save to historical storage
                    candle_data = {
                        'timestamp': current_timestamp,
                        'start_time': candle_start,
                        'end_time': candle_end,
                        'buy_volume': buy_volume,
                        'sell_volume': sell_volume,
                        'total_volume': total_volume,
                        'net_volume': buy_volume - sell_volume,
                        'avg_price': avg_price,
                        'close_price': close_price,
                        'max_buy_volume': max_buy_volume,
                        'max_buy_price': max_buy_price,
                        'max_sell_volume': max_sell_volume,
                        'max_sell_price': max_sell_price
                    }
                    
                    try:
                        historical_storage.save_whale_candle(symbol, tf, candle_data)
                    except Exception as e:
                        print(f"Error saving historical data for {symbol}/{tf}: {e}")
                
                # Generate and send report (unchanged)
                summary = generate_summary(
                    symbol_trades,
                    current_time,
                    symbol,
                    time_frame_config,
                    interval_seconds
                )
                
                if summary:
                    # Check if this report is different from the last one
                    report_hash = hash(summary)
                    last_report_time = last_reports.get(symbol, 0)
                    
                    if current_time - last_report_time > report_cooldown or report_hash != last_reports.get(f"{symbol}_hash"):
                        try:
                            await send_telegram_report(summary)
                            last_reports[symbol] = current_time
                            last_reports[f"{symbol}_hash"] = report_hash
                            consecutive_errors = 0
                            print(f"Report sent successfully for {symbol}\n")
                        except Exception as e:
                            consecutive_errors += 1
                            print(f"Telegram error ({consecutive_errors}): {e}")
                            
                            if consecutive_errors >= 3:
                                print("Too many errors, waiting 60 seconds...")
                                await asyncio.sleep(60)
            
            # Clean up old historical data periodically (once per day)
            if current_time - last_clean_time > 86400:
                try:
                    historical_storage.cleanup_old_data()
                except Exception as e:
                    print(f"Error cleaning up historical data: {e}")
            
            # Then send dollar volume reports for all configured timeframes
            if config.REPORT_DOLLAR_VOLUMES:
                for timeframe in config.DOLLAR_VOLUME_TIMEFRAMES:
                    dollar_data = generate_dollar_volume_data(timeframe, current_time)
                    if dollar_data:
                        # Calculate candle range for this specific timeframe
                        tf_config = get_time_frame_config().get(timeframe)
                        if tf_config:
                            seconds = tf_config[1]
                        else:
                            seconds = 900  # default to 15 minutes
            
                        candle_start, candle_end, trade_end = get_candle_range(
                            current_time, 
                            seconds,
                            interval_seconds
                        )
                        
                        # Save dollar volume data to historical storage
                        for coin, volumes in dollar_data.items():
                            candle_data = {
                                'timestamp': current_time,
                                'start_time': candle_start,
                                'end_time': candle_end,
                                'buy_dollar': volumes["buy"],
                                'sell_dollar': volumes["sell"],
                                'total_dollar': volumes["buy"] + volumes["sell"],
                                'net_dollar': volumes["buy"] - volumes["sell"],
                                'close_price': 0,  # Will be updated with actual close price
                                'balance_percent': volumes["buy"] / (volumes["buy"] + volumes["sell"]) * 100 if (volumes["buy"] + volumes["sell"]) > 0 else 0
                            }
                
                            # Try to get actual close price from trades
                            coin_trades = [t for t in all_significant_trades if t["coin"] == coin and candle_start <= t["timestamp"] < trade_end]
                            if coin_trades:
                                last_trade = max(coin_trades, key=lambda x: x["timestamp"])
                                candle_data['close_price'] = last_trade["price"]
                
                            try:
                                historical_storage.save_dollar_candle(coin, timeframe, candle_data)
                            except Exception as e:
                                print(f"Error saving dollar historical data for {coin}/{timeframe}: {e}")
                        
                        # Continue with existing report generation
                        dollar_reports = generate_detailed_dollar_report(
                            timeframe, 
                            dollar_data, 
                            current_time,
                            all_significant_trades
                        )
                        
                        # Check if this report is different from the last one
                        report_hash = hash(str(dollar_data))
                        last_report_time = last_reports.get(timeframe, 0)
                        
                        if current_time - last_report_time > report_cooldown or report_hash != last_reports.get(f"{timeframe}_hash"):
                            for report in dollar_reports:
                                if report.strip():  # Only send non-empty reports
                                    try:
                                        await send_telegram_report(report)
                                        print(f"Dollar volume report part sent for {timeframe}\n")
                                    except Exception as e:
                                        print(f"Error sending dollar volume report: {e}")
                            
                            last_reports[timeframe] = current_time
                            last_reports[f"{timeframe}_hash"] = report_hash
            
            # Calculate next report time
            next_report_time = current_time - (current_time % interval_seconds) + interval_seconds
            wait_time = next_report_time - current_time
            
            if wait_time > 0:
                await asyncio.sleep(wait_time)
                
        except Exception as e:
            print(f"Critical error in reporting loop: {e}")
            import traceback
            traceback.print_exc()
            print("Restarting reporting loop in 30 seconds...")
            await asyncio.sleep(30)

async def connect_to_websocket():
    global whale_trades, cum_d_volumes, stored_day
    global reporting_task
    global dollar_volume_data
    
    saved_state = load_state()
    current_time = get_ntp_time()
    
    time_frame_config = get_time_frame_config()
    
    existing_trades = load_existing_trades()
    whale_trades = existing_trades
    
    # Initialize last_save_time after loading existing trades
    last_save_time = time.time()
    
    default_cum_d = {symbol: 0.0 for symbol in config.SYMBOLS}
    
    if saved_state:
        print("Restoring from saved state")
        try:
            stored_day = saved_state.get("stored_day")
            
            saved_cum_d = saved_state.get("cum_d_volumes", {})
            for symbol in config.SYMBOLS:
                if symbol in saved_cum_d:
                    default_cum_d[symbol] = saved_cum_d[symbol]
            
            cum_d_volumes = default_cum_d
        except Exception as e:
            print(f"Error restoring state: {e}")
            cum_d_volumes = default_cum_d
    else:
        stored_day = datetime.utcfromtimestamp(current_time).strftime("%Y-%m-%d")
        cum_d_volumes = default_cum_d
    
    current_utc_day = datetime.utcfromtimestamp(current_time).strftime("%Y-%m-%d")
    day_start = datetime.strptime(current_utc_day, "%Y-%m-%d").timestamp()
    
    for symbol in config.SYMBOLS:
        cum_d = calculate_cumulative_volumes(
            whale_trades, 
            symbol,
            day_start
        )
        cum_d_volumes[symbol] = cum_d
    
    save_state()
    
    # NEW: Load dollar volume state from persistence ===================
    saved_dollar_state = historical_storage.load_dollar_volume_state()
    for timeframe in config.DOLLAR_VOLUME_TIMEFRAMES:
        if timeframe in saved_dollar_state:
            dollar_volume_data[timeframe] = saved_dollar_state[timeframe]
        else:
            dollar_volume_data[timeframe] = {"start_time": 0, "data": {}}
    # End of new ========================================================
    
    uri = "wss://api.hyperliquid.xyz/ws"
    reconnect_delay = 5  # Start with 5 seconds delay
    
    while True:
        try:
            async with websockets.connect(
                uri, 
                ping_interval=10,  # Reduced from 20 to 15
                ping_timeout=8,   # Reduced from 15 to 10
                close_timeout=5,
                max_queue=2048
            ) as websocket:
                print("WebSocket connected")
                reconnect_delay = 5  # Reset delay after successful connection
                
                if reporting_task is None or reporting_task.done():
                    reporting_task = asyncio.create_task(reporting_loop())
                
                # Subscribe to all priority coins for dollar volume reporting
                whale_coins = config.SYMBOLS
                dollar_coins = config.DOLLAR_SYMBOLS
                all_coins = list(set(whale_coins + dollar_coins + config.PRIORITY_COINS + config.MINI_PRIORITY_COINS))
                
                for symbol in all_coins:
                    await websocket.send(json.dumps({
                        "method": "subscribe",
                        "subscription": {"type": "trades", "coin": symbol}
                    }))
                    print(f"Subscribed to {symbol}")
                    await asyncio.sleep(0.1)
                
                while True:
                    try:
                        data = await asyncio.wait_for(websocket.recv(), timeout=20)
                        await process_trade_data(data)
                    except asyncio.TimeoutError:
                        # Send ping to keep connection alive
                        try:
                            await websocket.ping()
                        except:
                            break  # Break inner loop to reconnect
                    
        except Exception as e:
            print(f"WebSocket error: {e}")
            print(f"Reconnecting in {reconnect_delay} seconds...\n")
            await asyncio.sleep(reconnect_delay)
            
            # Exponential backoff for reconnect delay, max 60 seconds
            reconnect_delay = min(reconnect_delay * 2, 60)