import os
import sys
import asyncio
import time
import fcntl
from core.hyperliquid_ws import connect_to_websocket
from bot.telegram_bot import start_bot
from lib.config_manager import ConfigManager

# ======== Singleton Lock ========
LOCK_FILE = "/tmp/bot1.lock"

def acquire_lock():
    """File-based lock to prevent multiple instances"""
    lock_fd = os.open(LOCK_FILE, os.O_CREAT | os.O_RDWR, 0o644)
    try:
        fcntl.flock(lock_fd, fcntl.LOCK_EX | fcntl.LOCK_NB)
        return lock_fd
    except BlockingIOError:
        print("Error: Another bot instance is already running!")
        sys.exit(1)
    except Exception as e:
        print(f"Lock error: {str(e)}")
        sys.exit(1)

# ======== Main Function ========
async def main():
    # Validate initial configuration
    config = ConfigManager()
    
    if not config.TELEGRAM_TOKEN:
        print("❌ Telegram token not configured! Please set TELEGRAM_TOKEN in .env")
        return
    
    # Create storage directory
    os.makedirs("storage", exist_ok=True)
    
    # Run main tasks
    await asyncio.gather(
        start_bot(),
        connect_to_websocket()
    )

# ======== Entry Point ========
if __name__ == "__main__":
    # Enable ANSI color support on Windows
    if sys.platform == "win32":
        os.system("color")
    
    # Acquire singleton lock
    lock_fd = acquire_lock()
    
    try:
        while True:
            try:
                asyncio.run(main())
            except KeyboardInterrupt:
                print("🛑 Bot stopped by user")
                break
            except Exception as e:
                print(f"🔥 Critical error: {str(e)}")
                print("🔄 Restarting in 30 seconds...")
                time.sleep(30)
    finally:
        # Release lock on exit
        try:
            os.close(lock_fd)
            os.remove(LOCK_FILE)
        except Exception as e:
            print(f"⚠️ Lock cleanup error: {str(e)}")
