import os
import json
from dotenv import load_dotenv
from datetime import datetime

class ConfigManager:
    _instance = None
    DEFAULT_COLORS = ["blue", "green", "yellow", "cyan", "purple", "red", "white"]
    
    def __new__(cls):
        if cls._instance is None:
            cls._instance = super(ConfigManager, cls).__new__(cls)
            cls._instance.init()
        return cls._instance
    
    def init(self):
        load_dotenv()
        self.SYMBOLS = self._parse_list(os.getenv("SYMBOLS", "BTC"))
        self.THRESHOLDS = self._parse_thresholds()
        self.INTERVAL_MINUTES = int(os.getenv("INTERVAL_MINUTES", 5))
        self.TIME_FRAMES = os.getenv("TIME_FRAME", "15m,4h,1d").split(",")
        self.MAX_TRADE_AGE_DAYS = int(os.getenv("MAX_TRADE_AGE_DAYS", 30))
        self.TELEGRAM_TOKEN = os.getenv("TELEGRAM_TOKEN")
        self.CHAT_ID = os.getenv("CHAT_ID")
        
        # Automatic color assignment
        self.COIN_COLORS = self._parse_coin_colors()
        self._assign_automatic_colors()
    
    def _parse_list(self, value):
        items = []
        seen = set()
        for item in value.split(','):
            item = item.strip()
            if item and item not in seen:
                items.append(item)
                seen.add(item)
        return items
    
    def _parse_thresholds(self):
        thresholds = {}
        for item in os.getenv("THRESHOLDS", "BTC=1").split(","):
            if "=" in item:
                key, value = item.split("=")
                key = key.strip()
                if key:
                    thresholds[key] = float(value.strip())
        return thresholds
    
    def _parse_coin_colors(self):
        colors = {}
        coin_colors_config = os.getenv("COIN_COLORS", "")
        if coin_colors_config:
            for item in coin_colors_config.split(','):
                if ":" in item:
                    coin, color = item.split(":", 1)
                    coin = coin.strip()
                    color = color.strip()
                    if coin and color:
                        colors[coin] = color
        return colors
    
    def _assign_automatic_colors(self):
        """Assign automatic colors to symbols without manual color setting"""
        color_index = 0
        for symbol in self.SYMBOLS:
            if symbol not in self.COIN_COLORS:
                # Get next color in sequence
                color = self.DEFAULT_COLORS[color_index % len(self.DEFAULT_COLORS)]
                self.COIN_COLORS[symbol] = color
                color_index += 1
    
    def update_config(self, key, value):
        valid_keys = ["SYMBOLS", "THRESHOLDS", "COIN_COLORS", 
                     "INTERVAL_MINUTES", "TIME_FRAME", "MAX_TRADE_AGE_DAYS"]
        
        if key not in valid_keys:
            raise ValueError(f"Invalid config key: {key}")
        
        # Special handling for each key type
        if key == "INTERVAL_MINUTES":
            value = int(value)
            if not 1 <= value <= 60:
                raise ValueError("Interval must be between 1-60 minutes")
        
        # Update environment and reload config
        os.environ[key] = str(value)
        self.init()  # Reload configuration
        
        # Save changes to .env file
        self._save_to_dotenv(key, value)
        return f"✅ Config updated: {key} = {value}"
    
    def _save_to_dotenv(self, key, value):
        env_lines = []
        if os.path.exists(".env"):
            with open(".env", "r") as f:
                env_lines = f.readlines()
        
        updated = False
        for i, line in enumerate(env_lines):
            if line.startswith(f"{key}="):
                env_lines[i] = f"{key}={value}\n"
                updated = True
                break
        
        if not updated:
            env_lines.append(f"{key}={value}\n")
        
        with open(".env", "w") as f:
            f.writelines(env_lines)
    
    def get_config_summary(self):
        color_info = "\n".join(
            f"  - {coin}: {color}" 
            for coin, color in self.COIN_COLORS.items()
        )
        
        return (
            f"⚙️ Current Configuration:\n"
            f"• SYMBOLS: {', '.join(self.SYMBOLS)}\n"
            f"• THRESHOLDS: {self.THRESHOLDS}\n"
            f"• COIN_COLORS:\n{color_info}\n"
            f"• INTERVAL_MINUTES: {self.INTERVAL_MINUTES}\n"
            f"• TIME_FRAME: {', '.join(self.TIME_FRAMES)}\n"
            f"• MAX_TRADE_AGE_DAYS: {self.MAX_TRADE_AGE_DAYS}\n"
            f"• TELEGRAM_TOKEN: {'*****' if self.TELEGRAM_TOKEN else 'Not set'}\n"
            f"• CHAT_ID: {self.CHAT_ID or 'Not set'}"
        )