#-*- coding: utf-8 -*-
import asyncio
import telegram
from telegram import Update, BotCommand
from telegram.ext import (
    Application,
    CommandHandler,
    ContextTypes,
    MessageHandler,
    filters
)
from telegram.request import HTTPXRequest
from datetime import datetime
import pytz
import os
import json
from lib.config_manager import ConfigManager
from lib.historical_storage import HistoricalStorage
#from lib.historical_storage import historical_storage

UTC_TZ = pytz.utc

def set_historical_storage(storage_instance):
    global historical_storage
    historical_storage = storage_instance

#================= Logic of Evalution of Reports ===============
def get_volume_icon(current_volume, prev_volume):
    """Get icon for volume changes based on percentage change"""
    if prev_volume == 0:
        return "➖"
    change_percent = ((current_volume - prev_volume) / prev_volume) * 100
    if change_percent > 50:
        return "📈📈"  # High volume increase
    elif change_percent > 20:
        return "📈"    # Moderate volume increase
    elif change_percent < -50:
        return "📉📉"  # High volume decrease
    elif change_percent < -20:
        return "📉"    # Moderate volume decrease
    else:
        return "↔️"    # Volume relatively stable

def get_price_icon(current_price, prev_price):
    """Get icon for price changes based on percentage change"""
    if prev_price == 0:
        return "➖"
    change_percent = ((current_price - prev_price) / prev_price) * 100
    if change_percent > 1.0:
        return "🚀"
    elif change_percent < -1.0:
        return "📉"
    elif change_percent > 0.2:
        return "↗️"
    elif change_percent < -0.2:
        return "↘️"
    else:
        return "➖"

def calculate_buyer_power_index(buy_volume, sell_volume):
    """
    Calculate Buyer Power Index (BPI)
    Formula: (Buy - Sell) / (Buy + Sell)
    Returns value between -1 and 1
    """
    total = buy_volume + sell_volume
    if total == 0:
        return 0
    return (buy_volume - sell_volume) / total

def get_bpi_icon(bpi_value):
    """Get icon for BPI value"""
    if bpi_value >= 0.4:
        return "🚀"
    elif bpi_value >= 0.1:
        return "↗️"
    elif bpi_value >= -0.1:
        return "➖"
    elif bpi_value >= -0.3:
        return "↘️"
    else:
        return "📉"

def format_volume(value, unit="BTC"):
    """Format volume values with appropriate units"""
    if abs(value) >= 1000:
        return f"{value/1000:+.1f}K {unit}"
    else:
        return f"{value:+.1f} {unit}"

# ======= End of Logic ===============================================

def format_price(price):
    """
    Format price dynamically based on value with simple rules
    
    Args:
        price (float): The price value to format
        
    Returns:
        str: Formatted price string with appropriate decimals
    """
    if price is None or price == 0:
        return "N/A"
    
    # Dynamic formatting based on price value
    if price >= 1000:
        return f"{price:,.0f}"      # Above 1000: no decimals
    elif price >= 100:
        return f"{price:,.1f}"      # 100 to 1000: 1 decimal
    elif price >= 1:
        return f"{price:,.2f}"      # 1 to 100: 2 decimals
    elif price >= 0.1:
        return f"{price:,.3f}"      # 0.1 to 1: 3 decimals
    else:
        return f"{price:,.4f}"      # Below 0.1: 4 decimals


def get_time_frame_config():
    base_config = {
        "5m": ("5 Minutes", 300, "⚪"),
        "15m": ("15 Minutes", 900, "🔵"),
        "30m": ("30 Minutes", 1800, "🟣"),
        "45m": ("45 Minutes", 2700, "🟠"),
        "1h": ("1 Hour", 3600, "🟢"),
        "2h": ("2 Hours", 7200, "🟤"),
        "4h": ("4 Hours", 14400, "🟡"),
        "6h": ("6 Hours", 21600, "🟣"),
        "8h": ("8 Hours", 28800, "🟠"),
        "12h": ("12 Hours", 43200, "🟠"),
        "1d": ("1 Day", 86400, "🔴"),
        "3d": ("3 Days", 259200, "🟤"),
        "1w": ("1 Week", 604800, "🔵")
    }

    config = ConfigManager()
    time_frames = config.TIME_FRAMES
    result = {}

    for tf in time_frames:
        if tf in base_config:
            result[tf] = base_config[tf]
        else:
            try:
                if tf.endswith('m'):
                    minutes = int(tf[:-1])
                    seconds = minutes * 60
                    result[tf] = (f"{minutes} Minutes", seconds, "⚪")
                elif tf.endswith('h'):
                    hours = int(tf[:-1])
                    seconds = hours * 3600
                    result[tf] = (f"{hours} Hour{'s' if hours > 1 else ''}", seconds, "🟢")
                elif tf.endswith('d'):
                    days = int(tf[:-1])
                    seconds = days * 86400
                    result[tf] = (f"{days} Day{'s' if days > 1 else ''}", seconds, "🔴")
                elif tf.endswith('w'):
                    weeks = int(tf[:-1])
                    seconds = weeks * 604800
                    result[tf] = (f"{weeks} Week{'s' if weeks > 1 else ''}", seconds, "🔵")
            except ValueError:
                print(f"Warning: Invalid time frame: {tf}")

    return result

async def safe_send_message(bot, chat_id, text, max_retries=3):
    for attempt in range(max_retries):
        try:
            # Clean text from any problematic Markdown characters
            clean_text = text.replace("**", "*").replace("__", "_").replace("```", "`")
            
            await bot.send_message(
                chat_id=chat_id,
                text=clean_text,
                parse_mode="Markdown",
                disable_web_page_preview=True
            )
            return True
        except Exception as e:
            if attempt < max_retries - 1:
                wait_time = 5 * (attempt + 1)
                print(f"Telegram send error (attempt {attempt+1}): {str(e)}")
                print(f"Retrying in {wait_time} seconds...")
                await asyncio.sleep(wait_time)
            else:
                print(f"Failed to send after {max_retries} attempts")
                return False
    return False


async def send_telegram_report(summary):
    config = ConfigManager()
    if not config.TELEGRAM_TOKEN or not config.CHAT_ID:
        print("Telegram not configured")
        return

    try:
        bot = telegram.Bot(
            token=config.TELEGRAM_TOKEN,
            request=HTTPXRequest(connect_timeout=20.0, read_timeout=20.0)
        )
        success = await safe_send_message(bot, config.CHAT_ID, summary)

        if success:
            print("Report sent to Telegram")
        else:
            print("Failed to send report")
    except Exception as e:
        print(f"Telegram error: {str(e)}")


def get_candle_range(current_timestamp, seconds, reporting_interval):
    current_candle_end = ((current_timestamp // seconds) + 1) * seconds
    current_candle_start = current_candle_end - seconds

    prev_candle_end = current_candle_start
    prev_candle_start = prev_candle_end - seconds

    time_since_prev_candle_close = current_timestamp - prev_candle_end

    if current_timestamp >= prev_candle_end and time_since_prev_candle_close <= reporting_interval:
        return prev_candle_start, prev_candle_end, prev_candle_end
    else:
        return current_candle_start, current_candle_end, current_timestamp


def generate_detailed_dollar_report(timeframe="1h", dollar_volume_data=None, current_time=None, significant_trades=None):
    """Generate detailed dollar volume report for all coins with timestamp and price information per coin"""
    config = ConfigManager()
    
    if dollar_volume_data is None:
        dollar_volume_data = {}
    
    if significant_trades is None:
        significant_trades = []
    
    if not hasattr(config, 'REPORT_DOLLAR_VOLUMES') or not config.REPORT_DOLLAR_VOLUMES:
        return [""]
    
    if not dollar_volume_data:
        return [""]
    
    tf_config = get_time_frame_config().get(timeframe, (timeframe, 0, "⚪"))
    label = tf_config[0]
    seconds = tf_config[1]
    
    # Add timestamp to report
    if current_time is None:
        current_time = time.time()
    
    current_time_utc = datetime.fromtimestamp(current_time, tz=UTC_TZ)
    current_time_str = current_time_utc.strftime("%Y-%m-%d %H:%M:%S UTC")
    
    # Calculate candle range for this timeframe
    reporting_interval = config.INTERVAL_MINUTES * 60
    candle_start, candle_end, trade_end = get_candle_range(
        current_time, 
        seconds,
        reporting_interval
    )
    
    reports = []
    current_report = f"\n💵 **Dollar Volume Report ({label})** 📊\n"
    current_report += f"⏰ {current_time_str}\n"
    current_report += "─" * 60 + "\n"
    
    coin_data = []
    for coin, volumes in dollar_volume_data.items():
        net_volume = volumes["buy"] - volumes["sell"]
        total_volume = volumes["buy"] + volumes["sell"]
        
        if total_volume < config.BASE_DOLLAR_THRESHOLD:
            continue
        
        # Calculate average prices and close price for this coin
        avg_buy_price = 0
        avg_sell_price = 0
        close_price = 0
        
        # Get all trades for this coin in the timeframe
        coin_trades = [t for t in significant_trades if t["coin"] == coin and candle_start <= t["timestamp"] < trade_end]
        
        if coin_trades:
            # Separate buy and sell trades
            buy_trades = [t for t in coin_trades if t.get("side") == "B"]
            sell_trades = [t for t in coin_trades if t.get("side") != "B"]
            
            # Calculate average buy price
            if buy_trades:
                total_buy_value = sum(t["dollar_value"] for t in buy_trades)
                total_buy_volume = sum(t["volume"] for t in buy_trades)
                avg_buy_price = total_buy_value / total_buy_volume if total_buy_volume > 0 else 0
            
            # Calculate average sell price
            if sell_trades:
                total_sell_value = sum(t["dollar_value"] for t in sell_trades)
                total_sell_volume = sum(t["volume"] for t in sell_trades)
                avg_sell_price = total_sell_value / total_sell_volume if total_sell_volume > 0 else 0
            
            # Get close price (last trade price for this coin)
            if coin_trades:
                last_trade = max(coin_trades, key=lambda x: x["timestamp"])
                close_price = last_trade["price"]
        
        buy_sell_ratio = volumes["buy"] / volumes["sell"] if volumes["sell"] > 0 else float('inf')
        
        # Determine trend based on net volume
        if net_volume > 0:
            trend = "🚀"
        elif net_volume < 0:
            trend = "📉"
        else:
            trend = "⚪"
            
        # Get previous candle data for comparison
        prev_candle = historical_storage.get_previous_dollar_candle(coin, timeframe, candle_end)
        
        # Calculate previous buy/sell ratio
        prev_buy_sell_ratio = 0
        if prev_candle and prev_candle['sell_dollar'] > 0:
            prev_buy_sell_ratio = prev_candle['buy_dollar'] / prev_candle['sell_dollar']
        elif prev_candle:
            prev_buy_sell_ratio = float('inf')
        
        coin_data.append({
            'coin': coin,
            'buy': volumes["buy"],
            'sell': volumes["sell"],
            'net': net_volume,
            'total': total_volume,
            'ratio': buy_sell_ratio,
            'prev_ratio': prev_buy_sell_ratio,
            'trend': trend,
            'avg_buy_price': avg_buy_price,
            'avg_sell_price': avg_sell_price,
            'close_price': close_price,
            'prev_buy': prev_candle['buy_dollar'] if prev_candle else 0,
            'prev_sell': prev_candle['sell_dollar'] if prev_candle else 0,
            'prev_net': prev_candle['net_dollar'] if prev_candle else 0,
            'prev_total': prev_candle['total_dollar'] if prev_candle else 0,
            'prev_close': prev_candle['close_price'] if prev_candle else 0,
            'prev_balance': prev_candle['balance_percent'] if prev_candle else 0
        })
    
    if not coin_data:
        return [current_report + "No significant dollar volume activity.\n"]
    
    coin_data.sort(key=lambda x: abs(x['net']), reverse=True)
    
    def format_dollar(value):
        if abs(value) >= 1_000_000:
            return f"${value/1_000_000:+.1f}M"
        elif abs(value) >= 1_000:
            return f"${value/1_000:+.1f}K"
        else:
            return f"${value:+.0f}"
    
    # Add coins to report, splitting if necessary
    for i, coin in enumerate(coin_data):
        ratio_text = f"{coin['ratio']:.2f}" if coin['ratio'] != float('inf') else "∞"
        prev_ratio_text = f"{coin['prev_ratio']:.2f}" if coin['prev_ratio'] != float('inf') else "∞"
        
        # Calculate Buyer Power Index
        bpi_current = calculate_buyer_power_index(coin['buy'], coin['sell'])
        bpi_prev = calculate_buyer_power_index(coin['prev_buy'], coin['prev_sell'])
        bpi_icon = get_bpi_icon(bpi_current)
        
        # Get icons for volume and price
        volume_icon = get_volume_icon(coin['total'], coin['prev_total'])
        price_icon = get_price_icon(coin['close_price'], coin['prev_close'])
        
        coin_report = (
            f"{coin['trend']} **{coin['coin']}:**\n"
            f"├ Price: ${format_price(coin['close_price'])} {price_icon} (${format_price(coin['prev_close'])})\n"
            f"├ Buy: {format_dollar(coin['buy'])} (prev: {format_dollar(coin['prev_buy'])})\n"
            f"├ Sell: {format_dollar(coin['sell'])} (prev: {format_dollar(coin['prev_sell'])})\n"
            f"├ Vol: {format_dollar(coin['total'])} {volume_icon} ({format_dollar(coin['prev_total'])})\n"
            f"├ Net: {format_dollar(coin['net'])} {coin['trend']} (prev: {format_dollar(coin['prev_net'])})\n"
            f"├ BPI: {bpi_current:+.3f} {bpi_icon} (prev: {bpi_prev:+.3f})\n"
            f"├ Buy/Sell: {ratio_text} (prev: {prev_ratio_text})\n"
            f"└ Balance: {coin['buy']/coin['total']*100:.1f}% buy (prev: {coin['prev_balance']:.1f}%)\n"
        )
        
        # Check if adding this coin would exceed the limit
        if len(current_report) + len(coin_report) > 3800:
            reports.append(current_report)
            current_report = f"**{label} Report (Continued)**\n"
            current_report += "─" * 60 + "\n"
        
        current_report += coin_report
    
    # Add market summary to the last report
    total_buy = sum(coin['buy'] for coin in coin_data)
    total_sell = sum(coin['sell'] for coin in coin_data)
    total_volume = total_buy + total_sell
    total_net = total_buy - total_sell
    
    # Get previous market summary data
    prev_total_buy = sum(coin['prev_buy'] for coin in coin_data)
    prev_total_sell = sum(coin['prev_sell'] for coin in coin_data)
    prev_total_volume = prev_total_buy + prev_total_sell
    prev_total_net = prev_total_buy - prev_total_sell
    
    # Determine overall market trend
    if total_net > 0:
        market_trend = "🚀 Bullish"
    elif total_net < 0:
        market_trend = "📉 Bearish"
    else:
        market_trend = "⚪ Neutral"
    
    market_summary = (
        "─" * 60 + "\n"
        f"📈 **Market Summary:**\n"
        f"├ Total Buy: {format_dollar(total_buy)} ({format_dollar(prev_total_buy)})\n"
        f"├ Total Sell: {format_dollar(total_sell)} ({format_dollar(prev_total_sell)})\n"
        f"├ Total Vol: {format_dollar(total_volume)} ({format_dollar(prev_total_volume)})\n"
        f"├ Total Net: {format_dollar(total_net)} ({format_dollar(prev_total_net)})\n"
        f"└ Market Sentiment: {market_trend}\n"
    )
    
    if len(current_report) + len(market_summary) > 4096:
        reports.append(current_report)
        reports.append(market_summary)
    else:
        current_report += market_summary
        reports.append(current_report)
    
    return reports


def generate_summary(whale_trades, current_time, symbol, time_frame_config, reporting_interval):
    if not whale_trades:
        return None
    current_time_utc = datetime.fromtimestamp(current_time, tz=UTC_TZ)
    current_time_str = current_time_utc.strftime("%Y-%m-%d %H:%M:%S UTC")

    last_price = next((t["price"] for t in reversed(whale_trades) if "price" in t), 0)

    summary = f"🐋 **Whale Alert Report** 🚨\n"
    summary += f"⏰ {current_time_str} | {symbol}\n"
    summary += f"💰 Last Price: ${format_price(last_price)}\n\n"

    for tf, (label, seconds, bullet) in time_frame_config.items():
        current_timestamp = current_time_utc.timestamp()

        candle_start, candle_end, trade_end = get_candle_range(
            current_timestamp,
            seconds,
            reporting_interval
        )

        start_time = datetime.fromtimestamp(candle_start, tz=UTC_TZ)
        end_time = datetime.fromtimestamp(candle_end, tz=UTC_TZ)

        relevant_trades = [
            t for t in whale_trades
            if candle_start <= t["timestamp"] < trade_end
        ]

        buy_trades = [t for t in relevant_trades if t.get("side") == "B"]
        sell_trades = [t for t in relevant_trades if t.get("side") != "B"]

        buy_volume = sum(t["volume"] for t in buy_trades)
        sell_volume = sum(t["volume"] for t in sell_trades)
        total_volume = buy_volume + sell_volume

        avg_price = 0
        if total_volume > 0:
            avg_price = sum(t.get("price", 0) * t["volume"] for t in relevant_trades) / total_volume

        # NEW: Calculate close price (last trade in candle)
        close_price = 0
        if relevant_trades:
            last_trade = max(relevant_trades, key=lambda x: x["timestamp"])
            close_price = last_trade["price"]

        max_buy = {"volume": 0, "price": 0}
        if buy_trades:
            max_buy_trade = max(buy_trades, key=lambda x: x["volume"])
            max_buy = {"volume": max_buy_trade["volume"], "price": max_buy_trade["price"]}

        max_sell = {"volume": 0, "price": 0}
        if sell_trades:
            max_sell_trade = max(sell_trades, key=lambda x: x["volume"])
            max_sell = {"volume": max_sell_trade["volume"], "price": max_sell_trade["price"]}

        if buy_volume > sell_volume:
            trend = "More Buying 🚀"
        elif sell_volume > buy_volume:
            trend = "More Selling 📉"
        else:
            trend = "Balanced"

        # Get previous candle data for comparison
        prev_candle = historical_storage.get_previous_whale_candle(symbol, tf, candle_end)
        
        # Format previous values for display
        prev_avg = prev_candle['avg_price'] if prev_candle else 0
        prev_buy = prev_candle['buy_volume'] if prev_candle else 0
        prev_sell = prev_candle['sell_volume'] if prev_candle else 0
        prev_total = prev_candle['total_volume'] if prev_candle else 0
        prev_net = prev_candle['net_volume'] if prev_candle else 0
        prev_max_buy_vol = prev_candle['max_buy_volume'] if prev_candle else 0
        prev_max_buy_price = prev_candle['max_buy_price'] if prev_candle else 0
        prev_max_sell_vol = prev_candle['max_sell_volume'] if prev_candle else 0
        prev_max_sell_price = prev_candle['max_sell_price'] if prev_candle else 0
        prev_close = prev_candle['close_price'] if prev_candle and 'close_price' in prev_candle else 0  # NEW

        time_format = "%H:%M"
        if seconds >= 86400:
            time_format = "%Y-%m-%d"
        start_str = start_time.strftime(time_format)
        end_str = end_time.strftime(time_format)

        # Calculate Buyer Power Index
        bpi_current = calculate_buyer_power_index(buy_volume, sell_volume)
        bpi_prev = calculate_buyer_power_index(prev_buy, prev_sell)
        bpi_icon = get_bpi_icon(bpi_current)

        # Get icons for volume and close price
        volume_icon = get_volume_icon(total_volume, prev_total)
        price_icon = get_price_icon(close_price, prev_close)
        
        candle_summary = (
            f"{bullet} **{label}**\n"
            f"├ Candle: {start_str} to {end_str}\n"
            f"├ Price: ${format_price(close_price)} {price_icon} (${format_price(prev_close) if prev_close is not None else 'N/A'})\n"  # NEW
            f"├ Avg: ${format_price(avg_price)} (${format_price(prev_avg)})\n"
            f"├ Buy: {buy_volume:.1f} {symbol} (prev: {prev_buy:.1f})\n"
            f"├ Sell: {sell_volume:.1f} {symbol} (prev: {prev_sell:.1f})\n"
            f"├ Vol: {total_volume:.1f} {symbol} {volume_icon} ({prev_total:.1f})\n"
            f"├ Net: {buy_volume - sell_volume:+.1f} {symbol} (prev: {prev_net:+.1f})\n"
            f"├ BPI: {bpi_current:+.3f} {bpi_icon} (prev: {bpi_prev:+.3f})\n"
            f"├ Max Buy: {max_buy['volume']:.0f} {symbol} @ ${format_price(max_buy['price'])}\n"  # UPDATED
            f"└ Max Sell: {max_sell['volume']:.0f} {symbol} @ ${format_price(max_sell['price'])}\n\n"  # UPDATED
        )

        summary += candle_summary

    print("\n" + summary)
    return summary


async def setup_commands(application):
    commands = [
        BotCommand("start", "Start the bot"),
        BotCommand("config", "Show current configuration"),
        BotCommand("set", "Update configuration (e.g. /set INTERVAL_MINUTES=10)"),
        BotCommand("report", "Get instant report"),
        BotCommand("help", "Show help information"),
    ]
    await application.bot.set_my_commands(commands)


async def start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    user = update.effective_user
    await update.message.reply_text(
        f"🚀 Welcome {user.first_name}!\n"
        "HyperLiquid Whale Tracker Bot\n\n"
        "Available commands:\n"
        "/config - Show current settings\n"
        "/set - Update configuration\n"
        "/report - Get instant report"
    )


async def show_config(update: Update, context: ContextTypes.DEFAULT_TYPE):
    config = ConfigManager()
    await update.message.reply_text(config.get_config_summary())


async def update_config(update: Update, context: ContextTypes.DEFAULT_TYPE):
    try:
        config = ConfigManager()
        args = update.message.text.split(maxsplit=1)
        if len(args) < 2:
            await update.message.reply_text("Usage: /set KEY=VALUE")
            return
        key_value = args[1].split("=", 1)
        if len(key_value) != 2:
            await update.message.reply_text("Invalid format. Use KEY=VALUE")
            return

        key, value = key_value
        response = config.update_config(key.strip(), value.strip())
        await update.message.reply_text(response)
    except Exception as e:
        await update.message.reply_text(f"Error: {str(e)}")


async def instant_report(update: Update, context: ContextTypes.DEFAULT_TYPE):
    await update.message.reply_text("Fetching latest report...")
    await update.message.reply_text("Instant reports are not yet implemented")


async def help_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    help_text = (
        "🐋 HyperLiquid Whale Tracker Bot\n\n"
        "🔹 /start - Start the bot\n"
        "🔹 /config - Show current configuration\n"
        "🔹 /set KEY=VALUE - Update settings\n"
        "🔹 /report - Get instant report\n"
        "🔹 /help - Show this help\n\n"
        "📋 Configuration Examples:\n"
        "• /set SYMBOLS=BTC,ETH,SOL\n"
        "• /set THRESHOLDS=BTC=2,ETH=50,SOL=100\n"
        "• /set DOLLAR_SYMBOLS=LINK,PYTH,IO\n"
        "• /set INTERVAL_MINUTES=15\n"
        "• /set TIME_FRAME=15m,30m,1h,4h,1d\n"
        "• /set MAX_TRADE_AGE_DAYS=30\n"
        "• /set ADAPTIVE_THRESHOLD_ENABLED=True\n"
        "• /set BASE_DOLLAR_THRESHOLD=100000\n"
        "• /set HIGH_VOLATILITY_FACTOR=5.0\n"
        "• /set VOLATILITY_CHECK_INTERVAL=300\n"
        "• /set PRIORITY_COINS=BTC,ETH\n"
        "• /set MINI_PRIORITY_COINS=DOGE,TON,LINK\n"
        "• /set MINI_DOLLAR_THRESHOLD=10000\n"
        "• /set REPORT_DOLLAR_VOLUMES=True\n"
        "• /set DOLLAR_VOLUME_TIMEFRAMES=15m,1h,4h\n\n"
        "⚡ Pro Tips:\n"
        "- Separate multiple values with commas (no spaces)\n"
        "- Use True/False for boolean values\n"
        "- Time frames: m=minutes, h=hours, d=days, w=weeks\n"
        "- Dollar values should be in USD\n"
        "- Priority coins always show regardless of market conditions"
    )
    await update.message.reply_text(help_text)


async def start_bot():
    config = ConfigManager()
    if not config.TELEGRAM_TOKEN:
        print("Telegram token not configured!")
        return

    print("Starting Telegram bot...")

    application = Application.builder().token(config.TELEGRAM_TOKEN).request(
        HTTPXRequest(connect_timeout=20.0, read_timeout=20.0)
    ).build()

    application.add_handler(CommandHandler("start", start))
    application.add_handler(CommandHandler("config", show_config))
    application.add_handler(CommandHandler("set", update_config))
    application.add_handler(CommandHandler("report", instant_report))
    application.add_handler(CommandHandler("help", help_command))

    await setup_commands(application)

    await application.initialize()
    await application.start()
    await application.updater.start_polling()
    print("Telegram bot is running")

    while True:
        await asyncio.sleep(3600)