#!/bin/bash

# ==============================================
# Python Bot Manager - Production Ready (Fixed Version)
# ==============================================

# --------------------------
# Configuration
# --------------------------
APP_DIR="$HOME/public_html/bot1"
VENV_PYTHON="$HOME/virtualenv/public_html/bot1/3.11/bin/python3.11_bin"  # Fixed binary name
LOG_DIR="$APP_DIR/storage"
LOG_FILE="$LOG_DIR/bot.log"
ERROR_LOG_FILE="$LOG_DIR/error.log"
PID_FILE="$LOG_DIR/bot.pid"
APP_CMD="$VENV_PYTHON $APP_DIR/run.py"

# --------------------------
# Initialize Environment
# --------------------------
init_environment() {
    mkdir -p "$LOG_DIR"
    touch "$LOG_FILE" "$ERROR_LOG_FILE" "$PID_FILE"
    chmod 755 "$LOG_DIR"
    chmod 644 "$LOG_FILE" "$ERROR_LOG_FILE" "$PID_FILE"
}

# --------------------------
# Core Functions (Improved)
# --------------------------
start_bot() {
    # Check for existing processes
    if pgrep -f "$APP_CMD" >/dev/null; then
        local pids=$(pgrep -f "$APP_CMD" | tr '\n' ' ')
        echo "[ERROR] Bot already running (PIDs: $pids)"
        return 1
    fi
    
    # Start new instance
    echo "Starting bot process..."
    nohup $APP_CMD >> "$LOG_FILE" 2>> "$ERROR_LOG_FILE" &
    local pid=$!
    
    # Verify successful start
    sleep 3
    if ps -p $pid >/dev/null; then
        echo $pid > "$PID_FILE"
        echo "[SUCCESS] Bot started (PID: $pid)"
        return 0
    else
        echo "[ERROR] Process terminated immediately!"
        echo "Check error logs: $ERROR_LOG_FILE"
        tail -n 10 "$ERROR_LOG_FILE"
        return 1
    fi
}

stop_bot() {
    # Graceful termination using PID file
    if [ -f "$PID_FILE" ]; then
        local pid=$(cat "$PID_FILE")
        echo "Stopping bot (PID: $pid)..."
        
        kill -15 "$pid" 2>/dev/null
        sleep 5
        
        if ps -p "$pid" >/dev/null; then
            kill -9 "$pid"
            echo "[WARNING] Force killed PID: $pid"
        fi
        
        rm -f "$PID_FILE"
    fi
    
    # Cleanup any remaining processes
    pkill -f "$APP_CMD" && echo "Cleaned residual processes" || true
}

restart_bot() {
    stop_bot
    sleep 2
    start_bot
}

check_status() {
    local pids=$(pgrep -f "$APP_CMD")
    
    if [ -n "$pids" ]; then
        echo "Bot STATUS: RUNNING (PIDs: $(echo $pids | tr '\n' ' '))"
        return 0
    else
        echo "Bot STATUS: STOPPED"
        return 1
    fi
}

tail_log() {
    tail -f "$LOG_FILE"
}

show_help() {
    echo "Usage: $0 {start|stop|restart|status|log|help}"
    echo "Commands:"
    echo "  start    - Start bot instance"
    echo "  stop     - Stop running bot"
    echo "  restart  - Restart bot service"
    echo "  status   - Check bot status"
    echo "  log      - View live logs"
    echo "  help     - Show this help"
}

# --------------------------
# Main Execution
# --------------------------
init_environment

case "$1" in
    start|stop|restart|log)
        ${1}_bot
        ;;
    status)
        check_status
        ;;
    help|--help|-h)
        show_help
        ;;
    *)
        echo "Invalid command"
        show_help
        exit 1
        ;;
esac

exit $?
