import json
import os
import sqlite3
from datetime import datetime, timedelta
from lib.config_manager import ConfigManager

class HistoricalStorage:
    def __init__(self):
        self.config = ConfigManager()
        self.storage_dir = "/home/vabotsss/public_html/bot1/storage"
        self.db_path = os.path.join(self.storage_dir, "historical_data.db")
        self.init_database()
    
    def init_database(self):
        """Initialize SQLite database with required tables"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        # Table for whale trade candle data
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS whale_candles (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                symbol TEXT NOT NULL,
                timeframe TEXT NOT NULL,
                timestamp INTEGER NOT NULL,
                start_time INTEGER NOT NULL,
                end_time INTEGER NOT NULL,
                buy_volume REAL,
                sell_volume REAL,
                total_volume REAL,
                net_volume REAL,
                avg_price REAL,
                max_buy_volume REAL,
                max_buy_price REAL,
                max_sell_volume REAL,
                max_sell_price REAL,
                close_price REAL,
                UNIQUE(symbol, timeframe, end_time)
            )
        ''')
        
        # Check if close_price column exists, if not add it
        cursor.execute("PRAGMA table_info(whale_candles)")
        columns = [column[1] for column in cursor.fetchall()]
        if 'close_price' not in columns:
            cursor.execute('ALTER TABLE whale_candles ADD COLUMN close_price REAL')
            print("Added close_price column to whale_candles table")
        
        # Table for dollar volume candle data
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS dollar_candles (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                symbol TEXT NOT NULL,
                timeframe TEXT NOT NULL,
                timestamp INTEGER NOT NULL,
                start_time INTEGER NOT NULL,
                end_time INTEGER NOT NULL,
                buy_dollar REAL,
                sell_dollar REAL,
                total_dollar REAL,
                net_dollar REAL,
                close_price REAL,
                balance_percent REAL,
                UNIQUE(symbol, timeframe, end_time)
            )
        ''')
        
        # Indexes for faster queries
        cursor.execute('CREATE INDEX IF NOT EXISTS idx_whale_symbol_timeframe ON whale_candles(symbol, timeframe)')
        cursor.execute('CREATE INDEX IF NOT EXISTS idx_whale_end_time ON whale_candles(end_time)')
        cursor.execute('CREATE INDEX IF NOT EXISTS idx_dollar_symbol_timeframe ON dollar_candles(symbol, timeframe)')
        cursor.execute('CREATE INDEX IF NOT EXISTS idx_dollar_end_time ON dollar_candles(end_time)')
        
        conn.commit()
        conn.close()
    
    def save_whale_candle(self, symbol, timeframe, candle_data):
        """Save whale trade candle data"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
    
        cursor.execute('''
            INSERT OR REPLACE INTO whale_candles 
            (symbol, timeframe, timestamp, start_time, end_time, buy_volume, sell_volume, 
            total_volume, net_volume, avg_price, max_buy_volume, max_buy_price, 
            max_sell_volume, max_sell_price, close_price)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ''', (
            symbol, timeframe, candle_data['timestamp'], candle_data['start_time'],
            candle_data['end_time'], candle_data['buy_volume'], candle_data['sell_volume'],
            candle_data['total_volume'], candle_data['net_volume'], candle_data['avg_price'],
            candle_data['max_buy_volume'], candle_data['max_buy_price'],
            candle_data['max_sell_volume'], candle_data['max_sell_price'],
            candle_data['close_price']
        ))
    
        conn.commit()
        conn.close()
        print(f"Saved whale candle for {symbol}/{timeframe}: close_price={candle_data.get('close_price', 'N/A')}")
    
    def save_dollar_candle(self, symbol, timeframe, candle_data):
        """Save dollar volume candle data"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        cursor.execute('''
            INSERT OR REPLACE INTO dollar_candles 
            (symbol, timeframe, timestamp, start_time, end_time, buy_dollar, sell_dollar, 
             total_dollar, net_dollar, close_price, balance_percent)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ''', (
            symbol, timeframe, candle_data['timestamp'], candle_data['start_time'],
            candle_data['end_time'], candle_data['buy_dollar'], candle_data['sell_dollar'],
            candle_data['total_dollar'], candle_data['net_dollar'], candle_data['close_price'],
            candle_data['balance_percent']
        ))
        
        conn.commit()
        conn.close()
        print(f"Saved dollar candle for {symbol}/{timeframe} at {candle_data['end_time']}")
    
    def get_previous_whale_candle(self, symbol, timeframe, current_end_time):
        """Get previous whale candle data for comparison"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        # Find the candle that ended immediately before the current one
        cursor.execute('''
            SELECT * FROM whale_candles 
            WHERE symbol = ? AND timeframe = ? AND end_time < ?
            ORDER BY end_time DESC LIMIT 1
        ''', (symbol, timeframe, current_end_time))
        
        row = cursor.fetchone()
        conn.close()
        
        if row:
            # Handle case where close_price column might not exist yet
            close_price = row[15] if len(row) > 15 else None
            
            print(f"Retrieved previous candle for {symbol}/{timeframe}: close_price={close_price}")
            
            return {
                'timestamp': row[3],
                'buy_volume': row[6],
                'sell_volume': row[7],
                'total_volume': row[8],
                'net_volume': row[9],
                'avg_price': row[10],
                'max_buy_volume': row[11],
                'max_buy_price': row[12],
                'max_sell_volume': row[13],
                'max_sell_price': row[14],
                'close_price': close_price  # Use the value or default to 0
            }
        print(f"No previous candle found for {symbol}/{timeframe}")
        return None
    
    def get_previous_dollar_candle(self, symbol, timeframe, current_end_time):
        """Get previous dollar candle data for comparison"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        cursor.execute('''
            SELECT * FROM dollar_candles 
            WHERE symbol = ? AND timeframe = ? AND end_time < ?
            ORDER BY end_time DESC LIMIT 1
        ''', (symbol, timeframe, current_end_time))
        
        row = cursor.fetchone()
        conn.close()
        
        if row:
            return {
                'timestamp': row[3],
                'buy_dollar': row[6],
                'sell_dollar': row[7],
                'total_dollar': row[8],
                'net_dollar': row[9],
                'close_price': row[10],
                'balance_percent': row[11]
            }
        return None
    
    def cleanup_old_data(self, max_age_days=30):
        """Remove data older than specified days"""
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        cutoff_timestamp = (datetime.now() - timedelta(days=max_age_days)).timestamp()
        
        cursor.execute('DELETE FROM whale_candles WHERE timestamp < ?', (cutoff_timestamp,))
        cursor.execute('DELETE FROM dollar_candles WHERE timestamp < ?', (cutoff_timestamp,))
        
        conn.commit()
        conn.close()