# -*- coding: utf-8 -*-
import asyncio
import telegram
from telegram import Update, BotCommand
from telegram.ext import (
    Application,
    CommandHandler,
    ContextTypes,
    MessageHandler,
    filters
)
from datetime import datetime
import pytz
import os
import json
from lib.config_manager import ConfigManager

# تنظیمات منطقه زمانی
UTC_TZ = pytz.utc

# تابع پیکربندی تایم فریم‌ها
def get_time_frame_config():
    base_config = {
        "5m": ("5 Minutes", 300, "⚪"),
        "15m": ("15 Minutes", 900, "🔵"),
        "30m": ("30 Minutes", 1800, "🟣"),
        "1h": ("1 Hour", 3600, "🟢"),
        "4h": ("4 Hours", 14400, "🟡"),
        "12h": ("12 Hours", 43200, "🟠"),
        "1d": ("1 Day", 86400, "🔴"),
        "3d": ("3 Days", 259200, "🟤"),
        "1w": ("1 Week", 604800, "🔵")
    }
    
    config = {}
    time_frames = ConfigManager().TIME_FRAMES
    for tf in time_frames:
        if tf in base_config:
            config[tf] = base_config[tf]
        else:
            try:
                if tf.endswith('m'):
                    minutes = int(tf[:-1])
                    seconds = minutes * 60
                    config[tf] = (f"{minutes} Minutes", seconds, "⚪")
                elif tf.endswith('h'):
                    hours = int(tf[:-1])
                    seconds = hours * 3600
                    config[tf] = (f"{hours} Hour{'s' if hours > 1 else ''}", seconds, "🟢")
                elif tf.endswith('d'):
                    days = int(tf[:-1])
                    seconds = days * 86400
                    config[tf] = (f"{days} Day{'s' if days > 1 else ''}", seconds, "🔴")
                elif tf.endswith('w'):
                    weeks = int(tf[:-1])
                    seconds = weeks * 604800
                    config[tf] = (f"{weeks} Week{'s' if weeks > 1 else ''}", seconds, "🔵")
            except ValueError:
                print(f"⚠️ Invalid time frame: {tf}")
    
    return config

# تابع ایمن برای ارسال پیام با قابلیت تلاش مجدد
async def safe_send_message(bot, chat_id, text, max_retries=3):
    """ارسال ایمن پیام با مکانیزم تلاش مجدد"""
    for attempt in range(max_retries):
        try:
            await bot.send_message(
                chat_id=chat_id,
                text=text,
                parse_mode="Markdown",
                timeout=30  # افزایش زمان انتظار
            )
            return True
        except Exception as e:
            if attempt < max_retries - 1:
                wait_time = 5 * (attempt + 1)  # افزایش تدریجی زمان انتظار
                print(f"⚠️ Telegram send error (attempt {attempt+1}): {str(e)}")
                print(f"🔄 Retrying in {wait_time} seconds...")
                await asyncio.sleep(wait_time)
            else:
                print(f"❌ Failed to send after {max_retries} attempts")
                return False
    return False

# تابع اصلی ارسال گزارش به تلگرام
async def send_telegram_report(summary):
    """ارسال گزارش به تلگرام با مدیریت خطا"""
    config = ConfigManager()
    if not config.TELEGRAM_TOKEN or not config.CHAT_ID:
        print("❌ Telegram not configured")
        return
    
    try:
        bot = telegram.Bot(token=config.TELEGRAM_TOKEN)
        success = await safe_send_message(bot, config.CHAT_ID, summary)
        
        if success:
            print("\n✅ Report sent to Telegram")
        else:
            print("\n❌ Failed to send report")
    except Exception as e:
        print(f"\n❌ Telegram error: {str(e)}")

# تابع محاسبه محدوده زمانی کندل
def get_candle_range(current_timestamp, seconds, reporting_interval):
    """محاسبه محدوده زمانی برای کندلهای مختلف"""
    current_candle_end = ((current_timestamp // seconds) + 1) * seconds
    current_candle_start = current_candle_end - seconds
    
    prev_candle_end = current_candle_start
    prev_candle_start = prev_candle_end - seconds
    
    time_since_prev_candle_close = current_timestamp - prev_candle_end
    
    if current_timestamp >= prev_candle_end and time_since_prev_candle_close <= reporting_interval:
        return prev_candle_start, prev_candle_end, prev_candle_end
    else:
        return current_candle_start, current_candle_end, current_timestamp

# تابع تولید خلاصه گزارش
def generate_summary(whale_trades, current_time, symbol, time_frame_config, reporting_interval):
    """تولید گزارش خلاصه معاملات"""
    if not whale_trades:
        return None

    current_time_utc = datetime.fromtimestamp(current_time, tz=UTC_TZ)
    current_time_str = current_time_utc.strftime("%Y-%m-%d %H:%M:%S UTC")
    
    last_price = next((t["price"] for t in reversed(whale_trades) if "price" in t), 0)
    
    summary = f"🐋 **Whale Alert Report** 🚨\n"
    summary += f"⏰ {current_time_str} | {symbol}\n"
    summary += f"💰 Last Price: ${last_price:.5f}\n\n"

    for tf, (label, seconds, bullet) in time_frame_config.items():
        current_timestamp = current_time_utc.timestamp()
        
        candle_start, candle_end, trade_end = get_candle_range(
            current_timestamp, 
            seconds,
            reporting_interval
        )
        
        start_time = datetime.fromtimestamp(candle_start, tz=UTC_TZ)
        end_time = datetime.fromtimestamp(candle_end, tz=UTC_TZ)
        
        relevant_trades = [
            t for t in whale_trades 
            if candle_start <= t["timestamp"] < trade_end
        ]

        buy_trades = [t for t in relevant_trades if t.get("side") == "B"]
        sell_trades = [t for t in relevant_trades if t.get("side") != "B"]
        
        buy_volume = sum(t["volume"] for t in buy_trades)
        sell_volume = sum(t["volume"] for t in sell_trades)
        total_volume = buy_volume + sell_volume
        
        avg_price = 0
        if total_volume > 0:
            avg_price = sum(t.get("price", 0) * t["volume"] for t in relevant_trades) / total_volume
        
        max_buy = {"volume": 0, "price": 0}
        if buy_trades:
            max_buy_trade = max(buy_trades, key=lambda x: x["volume"])
            max_buy = {"volume": max_buy_trade["volume"], "price": max_buy_trade["price"]}
        
        max_sell = {"volume": 0, "price": 0}
        if sell_trades:
            max_sell_trade = max(sell_trades, key=lambda x: x["volume"])
            max_sell = {"volume": max_sell_trade["volume"], "price": max_sell_trade["price"]}
        
        if buy_volume > sell_volume:
            trend = "More Buying 🚀"
        elif sell_volume > buy_volume:
            trend = "More Selling 📉"
        else:
            trend = "Balanced"
        
        time_format = "%H:%M"
        if seconds >= 86400:
            time_format = "%Y-%m-%d"
        start_str = start_time.strftime(time_format)
        end_str = end_time.strftime(time_format)
        
        candle_summary = (
            f"{bullet} **{label}**\n"
            f"├ Candle: {start_str} to {end_str}\n"
            f"├ Avg: ${avg_price:.3f}\n"
            f"├ Buy: {buy_volume:.4f} {symbol}\n"
            f"├ Sell: {sell_volume:.4f} {symbol}\n"
            f"├ Net: {buy_volume - sell_volume:+.4f} {symbol}\n"
            f"├ Trend: {trend}\n"
            f"├ Max Buy: {max_buy['volume']:.4f} {symbol} @ ${max_buy['price']:.4f}\n"
            f"└ Max Sell: {max_sell['volume']:.4f} {symbol} @ ${max_sell['price']:.4f}\n\n"
        )
        
        summary += candle_summary

    print("\n" + summary)
    return summary

# تابع تنظیم دستورات تلگرام
async def setup_commands(application):
    """تنظیم دستورات تلگرام در منو"""
    commands = [
        BotCommand("start", "Start the bot"),
        BotCommand("config", "Show current configuration"),
        BotCommand("set", "Update configuration (e.g. /set INTERVAL_MINUTES=10)"),
        BotCommand("report", "Get instant report"),
        BotCommand("help", "Show help information"),
    ]
    await application.bot.set_my_commands(commands)

# تابع شروع گفتگو با ربات
async def start(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """خوشآمدگویی به کاربر جدید"""
    user = update.effective_user
    await update.message.reply_text(
        f"🚀 Welcome {user.first_name}!\n"
        "HyperLiquid Whale Tracker Bot\n\n"
        "Available commands:\n"
        "/config - Show current settings\n"
        "/set - Update configuration\n"
        "/report - Get instant report"
    )

# تابع نمایش تنظیمات جاری
async def show_config(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """نمایش تنظیمات فعلی"""
    config = ConfigManager()
    await update.message.reply_text(config.get_config_summary())

# تابع بهروزرسانی تنظیمات
async def update_config(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """بهروزرسانی تنظیمات"""
    try:
        config = ConfigManager()
        args = update.message.text.split(maxsplit=1)
        if len(args) < 2:
            await update.message.reply_text("❌ Usage: /set KEY=VALUE")
            return
        
        key_value = args[1].split("=", 1)
        if len(key_value) != 2:
            await update.message.reply_text("❌ Invalid format. Use KEY=VALUE")
            return
        
        key, value = key_value
        response = config.update_config(key.strip(), value.strip())
        await update.message.reply_text(response)
    except Exception as e:
        await update.message.reply_text(f"❌ Error: {str(e)}")

# تابع دریافت گزارش فوری
async def instant_report(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """ارسال گزارش فوری"""
    await update.message.reply_text("📊 Fetching latest report...")
    # TODO: Implement actual report generation
    await update.message.reply_text("⚠️ Instant reports are not yet implemented")

# تابع راهنمای دستورات
async def help_command(update: Update, context: ContextTypes.DEFAULT_TYPE):
    """نمایش راهنمای دستورات"""
    help_text = (
        "🐋 HyperLiquid Whale Tracker Bot\n\n"
        "🔹 /start - Start the bot\n"
        "🔹 /config - Show current configuration\n"
        "🔹 /set KEY=VALUE - Update settings\n"
        "🔹 /report - Get instant report\n"
        "🔹 /help - Show this help\n\n"
        "Example commands:\n"
        "/set INTERVAL_MINUTES=10\n"
        "/set SYMBOLS=BTC,ETH\n"
        "/set THRESHOLDS=BTC=1.5,ETH=10"
    )
    await update.message.reply_text(help_text)

# تابع اصلی راهاندازی ربات تلگرام
async def start_bot():
    """شروع و راهاندازی ربات تلگرام"""
    config = ConfigManager()
    
    if not config.TELEGRAM_TOKEN:
        print("❌ Telegram token not configured!")
        return
    
    print("🚀 Starting Telegram bot...")
    application = Application.builder().token(config.TELEGRAM_TOKEN).build()
    
    # Register commands
    application.add_handler(CommandHandler("start", start))
    application.add_handler(CommandHandler("config", show_config))
    application.add_handler(CommandHandler("set", update_config))
    application.add_handler(CommandHandler("report", instant_report))
    application.add_handler(CommandHandler("help", help_command))
    
    # Setup commands menu
    await setup_commands(application)
    
    # Start polling
    await application.initialize()
    await application.start()
    await application.updater.start_polling()
    print("✅ Telegram bot is running")
    
    # Keep running
    while True:
        await asyncio.sleep(3600)